#version 150
#extension GL_EXT_gpu_shader4 : enable
///////////////////////////////////////////////////////////////////////////////////////////////////
// iStripper wrapper for Shadertoy conversions by @Calgon  //
///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Notes
//
// Idea was to create a standard wrapper around Shadertoy code that could be applied to any shader
// sourced from Shadertoy.
// Version number is 150 as standard but where later functions are found this is changed to 330

// Wrapper Follows....
uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Declare the missing thingamies that aren't available on VGHD
uniform vec3  iChannelResolution[4];	// BUT WE NEED TO FILL THEM !
uniform float iChannelTime[4];

//#define iResolution u_WindowSize
#define iResolution vec3(u_WindowSize, 0)	// Because Shadertoy iResolution is a vec3
#define iFrame 0


//Slow the time functions down a little as standard
//#define iTime u_Elapsed * .5
float iTime;

#define iGlobalTime u_Elapsed * .5

// Seems the word texture is important and should not be replaced.  Therefore we must replace
// Shadertoy texture0..3 with texture 0..3 further down
uniform sampler2D texture0; //Random Surfaces
uniform sampler2D texture1; //Water
uniform sampler2D texture2; //Random Greys
uniform sampler2D texture3; //More surfaces

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture(sampler,fract(P),Bias);}

#define texture texture2D_Fract	// So whenever Shadertoy says "texture" we run it through this Macro
// Can we do the same for Cubemaps 

#define iChannel0 texture0
#define iChannel1 texture1
#define iChannel2 texture2
#define iChannel3 texture3

// Mouse Simulation from @TheEmu	
#define iMouse vec4(0.)
// Alternative Macro if iMouse is better moving
//#define iMouse AUTO_MOUSE  //vec4(0.0,0.0,0.0,0.0)
// Simple "Automatic Mouse". Simulates scanning the mouse over the full range of
// the screen with the X and Y scanning frequencies being different. TheEmu.
#define MOUSE_SPEED vec2(0.5,0.577777) * 0.2
//#define MOUSE_POS vec2((0.25+sin(iTime*MOUSE_SPEED*2))*u_WindowSize/2.0)
//#define MOUSE_POS vec2((sin(iTime*1)*.5*u_WindowSize.x/2.0),1.0*u_WindowSize.y/2.0)
//#define MOUSE_PRESS vec2(0.0,0.0)
#define MOUSE_PRESS vec2(1.0,1.0)
#define AUTO_MOUSE vec4( MOUSE_POS, MOUSE_PRESS )



vec4 iDate;


///////////////////////////////////////////////////////////////////////////////////////////////////
// Uniforms to control timing of each shader
//uniform float cycle_time;
uniform float bgnum;
uniform float maxbgnum;
//uniform float alpha_off;
//uniform float alpha_on;
//uniform float alpha_always_on;
uniform float scene_duration;


///////////////////////////////////////////////////////////////////////////////////////////////////
// Extract a pixel from texture1 to get the random number
float randNum = texture(texture2, vec2(0.0, 0.0)).r;
float seed_start = 3*randNum-1.5;



///////////////////////////////////////////////////////////////////////////////////////////////////
// Shadertoy Code follows
///////////////////////////////////////////////////////////////////////////////////////////////////

// Sourced from https://www.shadertoy.com/view/wldXzX
#define TOY
#define PI   3.141592653589
#define PI2  1.570796326795
#define TAU  6.283185307178
#define E    2.718281828459
#define EPS  0.000000000001
#define PHI  1.618033988750
#define EPS1 1.00001

#define KEY_LEFT  37
#define KEY_UP    38
#define KEY_RIGHT 39
#define KEY_DOWN  40
#define KEY_SPACE 32
#define KEY_1     49
#define KEY_9     57
#define KEY_A     65
#define KEY_C     67
#define KEY_D     68
#define KEY_E     69
#define KEY_F     70
#define KEY_N     78
#define KEY_Q     81
#define KEY_R     82
#define KEY_S     83
#define KEY_W     87
#define KEY_X     88
#define KEY_Z     90

const vec3 v0 = vec3(0,0,0);
const vec3 vx = vec3(1,0,0);
const vec3 vy = vec3(0,1,0);
const vec3 vz = vec3(0,0,1);

const vec3 red   = vec3(0.8,0.0,0.0);
const vec3 green = vec3(0.0,0.5,0.0);
const vec3 blue  = vec3(0.2,0.2,1.0);
const vec3 white = vec3(1.0,1.0,1.0);
const vec3 black = vec3(0.0,0.0,0.0);

#define sdMat(m,d)  if (d < gl.sdf.dist) { gl.sdf.dist = d; gl.sdf.mat = m; }
    
//  0000000   000       0000000   0000000     0000000   000      
// 000        000      000   000  000   000  000   000  000      
// 000  0000  000      000   000  0000000    000000000  000      
// 000   000  000      000   000  000   000  000   000  000      
//  0000000   0000000   0000000   0000000    000   000  0000000  

struct Text {
    ivec2 size;
    ivec2 adv;
} text;

struct SDF {
    float dist;
    vec3  pos;
    int   mat;
};

struct _gl {
    vec2  uv;
    vec2  frag;
    vec2  mouse;
    vec2  mp;
    ivec2 ifrag;
    float aspect;
    vec4  color;
    int   option;
    float time;
    vec3  light1;
    vec3  light2;
    vec3  light3;
    vec3  rd;
    float ambient;
    float shadow;
    int   zero;
    bool  march;
    SDF   sdf;
} gl;

struct _cam {
    vec3  tgt;
    vec3  pos;
    vec3  pos2tgt;
    vec3  dir;
    vec3  up;
    vec3  x;
    float dist;
    float fov;
} cam;

struct Mat {
    float hue;
    float sat;
    float lum;
    float shiny;
    float glossy;
};

uniform sampler2D fontChannel;

void initGlobal(vec2 fragCoord, vec3 resolution, vec4 mouse, float time)
{
    text.size = ivec2(16,32)*2;
    text.adv  = ivec2(text.size.x,0);
    
    mouse.xy = min(mouse.xy,resolution.xy);
    if (mouse.z < 1.0)
    {
        if (mouse.z > -1.0)
            gl.mouse = resolution.xy*0.5;
        else
            gl.mouse = mouse.xy;
    }
    else gl.mouse = mouse.xy;
    
    gl.mp = (2.0*abs(gl.mouse)-vec2(resolution.xy))/resolution.y;    

    gl.aspect = resolution.x / resolution.y;
    gl.frag   = fragCoord;
    gl.ifrag  = ivec2(fragCoord);
    gl.uv     = (fragCoord+fragCoord-resolution.xy)/resolution.y;
    
    gl.ambient = 0.03;
    gl.shadow  = 0.20;
}

float powi(int a, int b) { return pow(float(a), float(b)); }
float log10(float a) { return log(a)/log(10.0); }
float clamp01(float v) { return clamp(v, 0.0, 1.0); }
vec3  clamp01(vec3 v) { return clamp(v, 0.0, 1.0); }

// 00000000   00000000   000  000   000  000000000  
// 000   000  000   000  000  0000  000     000     
// 00000000   0000000    000  000 0 000     000     
// 000        000   000  000  000  0000     000     
// 000        000   000  000  000   000     000     

#ifndef TOY
float print(ivec2 pos, int ch)
{
    ivec2 r = gl.ifrag-pos; bool i = r.y>0 && r.x>0 && r.x<=text.size.y && r.y<=text.size.y;
    return i ? texelFetch(iChannel2,ivec2((ch%16)*64,(1024-64-64*(ch/16)))+r*64/text.size.y,0).r : 0.0;
}

float print(ivec2 pos, float v)
{
    float c = 0.0; ivec2 a = text.adv; 
    float fv = fract(v);
    v = (fv > 0.995 || fv < 0.005) ? round(v) : v;
    float f = abs(v);
    int i = (fv == 0.0) ? 1 : fract(v*10.0) == 0.0 ? -1 : -2;
    int ch, u = max(1,int(log10(f))+1);
    ivec2 p = pos+6*a;
    for (; i <= u; i++) {
        if (i == 0)     ch = 46;
        else if (i > 0) ch = 48+int(mod(f, powi(10,i))/powi(10,i-1));
        else            ch = 48+int(mod(f+0.005, powi(10,i+1))/powi(10,i));
        c = max(c, print(p-i*a, ch)*float(i+3)/30.0); }
    if (v < 0.0) c = max(c, print(p-i*a, 45)*float(i)/30.0);
    return c;
}

float print(ivec2 pos, vec4 v)
{
    float c = 0.0;
    for (int i = 0; i < 4; i++) {
        c = max(c, print(pos, v[i]));
        pos += text.adv*8; }
    return c;
}

float print(ivec2 pos, vec3 v)
{
    float c = 0.0;
    for (int i = 0; i < 3; i++) {
        c = max(c, print(pos, v[i]));
        pos += text.adv*8; }
    return c;
}

float print(ivec2 pos, vec2 v)
{
    float c = 0.0;
    for (int i = 0; i < 2; i++) {
        c = max(c, print(pos, v[i]));
        pos += text.adv*8; }
    return c;
}

float print(int x, int y, int v)   { return print(ivec2(text.size.x*x,text.size.y*y), float(v)); }
float print(int x, int y, float v) { return print(ivec2(text.size.x*x,text.size.y*y), v); }
float print(int x, int y, vec4 v)  { return print(ivec2(text.size.x*x,text.size.y*y), v); }
float print(int x, int y, vec3 v)  { return print(ivec2(text.size.x*x,text.size.y*y), v); }
float print(int x, int y, vec2 v)  { return print(ivec2(text.size.x*x,text.size.y*y), v); }
float print(int x, int y, ivec3 v) { return print(ivec2(text.size.x*x,text.size.y*y), vec3(v)); }
#endif

// 000   000   0000000    0000000  000   000  
// 000   000  000   000  000       000   000  
// 000000000  000000000  0000000   000000000  
// 000   000  000   000       000  000   000  
// 000   000  000   000  0000000   000   000  

float hash11(float p)
{
    p = fract(p * 0.1031);
    p *= p + 33.33;
    p *= p + p;
    return fract(p);
}

vec3 hash33(vec3 p3)
{
    p3 = fract(p3 * vec3(12.3,456.7,8912.3));
    p3 += dot(p3, p3.yxz+33.33);
    return fract((p3.xxy + p3.yxx)*p3.zyx);
}

vec3 hash31(float p)
{
   return hash33(vec3(p));
}

float hash12(vec2 p)
{
    vec3 p3  = fract(vec3(p.xyx) * .1031);
    p3 += dot(p3, p3.yzx + 33.33);
    return fract((p3.x + p3.y) * p3.z);
}

float gradientNoise(vec2 v)
{
    return fract(52.9829189 * fract(dot(v, vec2(0.06711056, 0.00583715))));
}

// 000   000   0000000  000      
// 000   000  000       000      
// 000000000  0000000   000      
// 000   000       000  000      
// 000   000  0000000   0000000  

vec3 hsl2rgb( in vec3 c )
{
    vec3 rgb = clamp( abs(mod(c.x*6.0+vec3(0.0,4.0,2.0),6.0)-3.0)-1.0, 0.0, 1.0 );
    return c.z + c.y * (rgb-0.5)*(1.0-abs(2.0*c.z-1.0));
}

vec3 hsl(float h, float s, float l) { return hsl2rgb(vec3(h,s,l)); }

vec3 rgb2hsl( vec3 col )
{
    float minc = min( col.r, min(col.g, col.b) );
    float maxc = max( col.r, max(col.g, col.b) );
    vec3  mask = step(col.grr,col.rgb) * step(col.bbg,col.rgb);
    vec3 h = mask * (vec3(0.0,2.0,4.0) + (col.gbr-col.brg)/(maxc-minc + EPS)) / 6.0;
    return vec3( fract( 1.0 + h.x + h.y + h.z ),              
                 (maxc-minc)/(1.0-abs(minc+maxc-1.0) + EPS),  
                 (minc+maxc)*0.5);
}

vec3 colsat(vec3 col, float sat)
{
    vec3 h = rgb2hsl(col);
    return hsl(h.x,sat,h.z);
}

vec3 gray(vec3 col)
{
    return colsat(col, 0.0);
}

// 00     00   0000000   000000000  00000000   000  000   000  
// 000   000  000   000     000     000   000  000   000 000   
// 000000000  000000000     000     0000000    000    00000    
// 000 0 000  000   000     000     000   000  000   000 000   
// 000   000  000   000     000     000   000  000  000   000  

mat3 alignMatrix(vec3 dir) 
{
    vec3 f = normalize(dir);
    vec3 s = normalize(cross(f, vec3(0.48, 0.6, 0.64)));
    vec3 u = cross(s, f);
    return mat3(u, s, f);
}

// 00000000    0000000   000000000  
// 000   000  000   000     000     
// 0000000    000   000     000     
// 000   000  000   000     000     
// 000   000   0000000      000     

float rad2deg(float r) { return 180.0 * r / PI; }
float deg2rad(float d) { return PI * d / 180.0; }

vec3  rad2deg(vec3 v) { return 180.0 * v / PI; }
vec3  deg2rad(vec3 v) { return PI * v / 180.0; }

mat3  rotMat(vec3 u, float angle)
{
    float s = sin(deg2rad(angle));
    float c = cos(deg2rad(angle));
    float i = 1.0-c;
    
    return mat3(
        c+u.x*u.x*i, u.x*u.y*i-u.z*s, u.x*u.z*i+u.y*s,
        u.y*u.x*i+u.z*s, c+u.y*u.y*i, u.y*u.z*i-u.x*s,
        u.z*u.x*i-u.y*s, u.z*u.y*i+u.x*s, c+u.z*u.z*i
        );
}

vec3 rotAxisAngle(vec3 position, vec3 axis, float angle)
{
    mat3 m = rotMat(axis, angle);
    return m * position;
}

// 00000000    0000000   000       0000000   00000000   
// 000   000  000   000  000      000   000  000   000  
// 00000000   000   000  000      000000000  0000000    
// 000        000   000  000      000   000  000   000  
// 000         0000000   0000000  000   000  000   000  

vec3 polar(vec3 v)
{
    float radius = length(v);
    float phi    = atan(v.y, v.x);
    float rho    = acos(v.z/radius);
    return vec3(phi, rho, radius);
}

vec3 unpolar(vec3 v)
{
    float s = sin(v.y);
    float x = s * cos(v.x);
    float y = s * sin(v.x);
    float z =     cos(v.y);
    return vec3(x, y, z)*v.z;
}

vec3 polar2(vec3 v)
{
    float radius = length(v);
    float phi    = atan(v.z, v.x);
    float rho    = acos(v.y/radius);
    return vec3(phi, rho, radius);
}

vec3 unpolar2(vec3 v)
{
    float s = sin(v.y);
    float x = s * cos(v.x);
    float z = s * sin(v.x);
    float y =     cos(v.y);
    return vec3(x, y, z)*v.z;
}

//  0000000   000   000   0000000   000000000  
// 000   000  000   000  000   000     000     
// 000 00 00  000   000  000000000     000     
// 000 0000   000   000  000   000     000     
//  00000 00   0000000   000   000     000     

vec4 quatAxisAngle(vec3 axis, float angle)
{ 
    float half_angle = deg2rad(angle*0.5);
    return vec4(axis*sin(half_angle), cos(half_angle));
}

vec4 quatConj(vec4 q)
{ 
    return vec4(-q.x, -q.y, -q.z, q.w); 
}
  
vec4 quatMul(vec4 q1, vec4 q2)
{ 
    vec4 qr;
    qr.x = (q1.w * q2.x) + (q1.x * q2.w) + (q1.y * q2.z) - (q1.z * q2.y);
    qr.y = (q1.w * q2.y) - (q1.x * q2.z) + (q1.y * q2.w) + (q1.z * q2.x);
    qr.z = (q1.w * q2.z) + (q1.x * q2.y) - (q1.y * q2.x) + (q1.z * q2.w);
    qr.w = (q1.w * q2.w) - (q1.x * q2.x) - (q1.y * q2.y) - (q1.z * q2.z);
    return qr;
}

vec3 rotAxisAngleQuat(vec3 p, vec3 axis, float angle)
{ 
    vec4 qr = quatAxisAngle(axis, angle);
    return quatMul(quatMul(qr, vec4(p, 0)), quatConj(qr)).xyz;
}

vec3 rotRayAngle(vec3 p, vec3 ro, vec3 rd, float angle)
{ 
    return rotAxisAngle(p-ro, rd-ro, angle)+ro;
}

vec3 rotY(vec3 v, float d)
{
    float r = deg2rad(d);
    float c = cos(r);
    float s = sin(r);
    return vec3(v.x*c+v.z*s, v.y, v.z*c+v.x*s);
}

vec3 rotX(vec3 v, float d)
{
    float r = deg2rad(d);
    float c = cos(r);
    float s = sin(r);
    return vec3(v.x, v.y*c+v.z*s, v.z*c+v.y*s);
}

vec3 rotZ(vec3 v, float d)
{
    float r = deg2rad(d);
    float c = cos(r);
    float s = sin(r);
    return vec3(v.x*c+v.y*s, v.y*c+v.x*s, v.z);
}

//  0000000   00000000   0000000   00     00    
// 000        000       000   000  000   000    
// 000  0000  0000000   000   000  000000000    
// 000   000  000       000   000  000 0 000    
//  0000000   00000000   0000000   000   000    

vec3 posOnPlane(vec3 p, vec3 a, vec3 n)
{
    return p-dot(p-a,n)*n;
}

vec3 posOnRay(vec3 ro, vec3 rd, vec3 p)
{
    return ro + max(0.0, dot(p - ro, rd) / dot(rd, rd)) * rd;
}

bool rayIntersectsSphere(vec3 ro, vec3 rd, vec3 ctr, float r)
{
    return length(posOnRay(ro, rd, ctr) - ctr) < r;
}

//  0000000   00000000   
// 000   000  000   000  
// 000   000  00000000   
// 000   000  000        
//  0000000   000        

float opUnion(float d1, float d2, float k)
{
    float h = clamp(0.5 + 0.5*(d2-d1)/k, 0.0, 1.0);
    return mix(d2, d1, h) - k*h*(1.0-h);
}

float opDiff(float d1, float d2, float k) 
{
    float h = clamp(0.5 - 0.5*(d2+d1)/k, 0.0, 1.0);
    return mix(d1, -d2, h) + k*h*(1.0-h);
}

float opInter(float d1, float d2, float k) 
{
    
    float h = clamp(0.5 - 0.5*(d2-d1)/k, 0.0, 1.0);
    return mix(d2, d1, h) + k*h*(1.0-h);
}

float opDiff (float d1, float d2) { return opDiff (d1, d2, 0.0); }
float opUnion(float d1, float d2) { return opUnion(d1, d2, 0.5); }
float opInter(float d1, float d2) { return opInter(d1, d2, 0.2); }

//  0000000  0000000    
// 000       000   000  
// 0000000   000   000  
//      000  000   000  
// 0000000   0000000    

float sdSphere(vec3 a, float r)
{
    return length(gl.sdf.pos-a)-r;
}

float sdPill(vec3 a, float r, vec3 n)
{
    vec3 p = gl.sdf.pos-a;
    float d = abs(dot(normalize(n),normalize(p)));
    float f = smoothstep(0.0, 1.3, d);
    return length(p) - r + f * length(n);
}

float sdPlane(vec3 a, vec3 n)
{   
    return dot(n, gl.sdf.pos-a);
}

float sdPlane(vec3 n)
{   
    return dot(n, gl.sdf.pos);
}

float sdBox(vec3 a, vec3 b, float r)
{
  vec3 q = abs(gl.sdf.pos-a)-b;
  return length(max(q,0.0)) + min(max(q.x,max(q.y,q.z)),0.0) - r;
}

float sdEllipsoid(vec3 a, vec3 r)
{
    vec3 p = gl.sdf.pos-a;
    float k0 = length(p/r);
    float k1 = length(p/(r*r));
    return k0*(k0-1.0)/k1;
}

float sdCone(vec3 a, vec3 b, float r1, float r2)
{
    vec3 ab = b-a;
    vec3 ap = gl.sdf.pos-a;
    float t = dot(ab,ap) / dot(ab,ab);
    t = clamp(t, 0.0, 1.0);
    vec3 c = a + t*ab;
    return length(gl.sdf.pos-c)-(t*r2+(1.0-t)*r1);      
}

float sdLine(vec3 a, vec3 n, float r)
{
    vec3 p = gl.sdf.pos-a;
    return length(p-n*dot(p,n))-r;
}

float sdLine(vec2 p, vec2 a, vec2 b)
{
    vec2 n = b-a;
    vec2 nc = n.yx; nc.x *= -1.0;
    return dot(p-a,nc) <= 0.0 ? 0.0 : length((p-a)-n*dot(p-a,n)/dot(n,n));
}

float sdLine2(vec2 p, vec2 a, vec2 b)
{
    vec2 n = b-a;
    return length((p-a)-n*dot(p-a,n)/dot(n,n));
}

float sdCapsule(vec3 a, vec3 b, float r)
{
    vec3 ab = b-a;
    vec3 ap = gl.sdf.pos-a;
    float t = dot(ab,ap) / dot(ab,ab);
    t = clamp(t, 0.0, 1.0);
    vec3 c = a + t*ab;
    return length(gl.sdf.pos-c)-r;        
}

float sdCylinder(vec3 a, vec3 b, float r, float cr)
{
  vec3  ba = b - a;
  vec3  pa = gl.sdf.pos - a;
  float baba = dot(ba,ba);
  float paba = dot(pa,ba);
  float x = length(pa*baba-ba*paba) - r*baba;
  float y = abs(paba-baba*0.5)-baba*0.5;
  float x2 = x*x;
  float y2 = y*y*baba;
  float d = (max(x,y)<0.0)?-min(x2,y2):(((x>0.0)?x2:0.0)+((y>0.0)?y2:0.0));
  return sign(d)*sqrt(abs(d))/baba - cr;
}

float sdHexagon(vec3 p, vec3 a, vec3 r) // r: (radius, height, bevel)
{
    vec3 k = vec3(-0.8660254, 0.5, 0.57735);
    p = abs(p - a);
    p.xz -= 2.0*min(dot(k.xy, p.xz), 0.0)*k.xy;
    float hr = r.x-r.z;
    float hh = r.y-r.z;
    vec2 d = vec2(length(p.xz-vec2(clamp(p.x,-k.z*hr,k.z*hr), hr))*sign(p.z-hr),
p.y-hh);
    return min(max(d.x,d.y),0.0) + length(max(d,0.0)) - r.z;
}

float sdHexagon(vec3 a, vec3 r) // r: (radius, height, bevel)
{
    return sdHexagon(gl.sdf.pos, a, r);
}

vec3 posOnPlane(vec3 p, vec3 n)
{
    return p-dot(p,n)*n;
}

float sdTorus(vec3 p, vec3 a, vec3 n, float rl, float rs)
{
    vec3 q = p-a;
    return length(vec2(length(posOnPlane(q, n))-rl,abs(dot(n, q))))-rs;
}

// 000   000   0000000   000   0000000  00000000  
// 0000  000  000   000  000  000       000       
// 000 0 000  000   000  000  0000000   0000000   
// 000  0000  000   000  000       000  000       
// 000   000   0000000   000  0000000   00000000  

float noise3D(in vec3 p)
{
    const vec3 s = vec3(7, 157, 113);
    vec3 ip = floor(p); p -= ip; 
    vec4 h = vec4(0., s.yz, s.y + s.z) + dot(ip, s);
    p = p*p*(3. - 2.*p); 
    h = mix(fract(sin(h)*43758.5453), fract(sin(h + s.x)*43758.5453), p.x);
    h.xy = mix(h.xz, h.yw, p.y);
    return mix(h.x, h.y, p.z);
}

// 0000000    000   000  00     00  00000000   
// 000   000  000   000  000   000  000   000  
// 0000000    000   000  000000000  00000000   
// 000   000  000   000  000 0 000  000        
// 0000000     0000000   000   000  000        

float drawSphere(in vec3 p)
{
    p = fract(p)-.5; return dot(p, p);
}

float cellTile(in vec3 p)
{
    vec4 d; 
    d.x = drawSphere(p - vec3(.81, .62, .53)); p.xy = vec2(p.y-p.x, p.y + p.x)*.7071;
    d.y = drawSphere(p - vec3(.39, .2,  .11)); p.yz = vec2(p.z-p.y, p.z + p.y)*.7071;
    d.z = drawSphere(p - vec3(.62, .24, .06)); p.xz = vec2(p.z-p.x, p.z + p.x)*.7071;
    d.w = drawSphere(p - vec3(.2,  .82, .64));
    d.xy = min(d.xz, d.yw);
    return min(d.x, d.y)*2.66;
}

float bumpSurf(vec3 p)
{
    return 0.2*noise3D(p*15.0) - 0.05*noise3D(p*120.0);
}

vec3 bumpMap(vec3 p, vec3 nor, float factor)
{
    const vec2 e = vec2(0.001, 0);
    vec3 grad = (vec3(bumpSurf(p - e.xyy),
                      bumpSurf(p - e.yxy),
                      bumpSurf(p - e.yyx))-bumpSurf(p))/e.x;                     
    grad -= nor*dot(nor, grad);          
    return normalize(nor - grad*factor);
}

// 0000000     0000000    0000000  000   0000000  
// 000   000  000   000  000       000  000       
// 0000000    000000000  0000000   000  0000000   
// 000   000  000   000       000  000       000  
// 0000000    000   000  0000000   000  0000000   

void basis(vec3 n, out vec3 right, out vec3 front) 
{
    if (n.y < -0.999999)
    {
        right = -vz;
        front = -vx;
    } 
    else 
    {
        float a = 1.0/(1.0+n.y);
        float b = -n.x*n.z*a;
        right = vec3(1.0-n.x*n.x*a,-n.x,b);
        front = vec3(b,-n.z,1.0-n.z*n.z*a);
    }
}

//  0000000   0000000   00     00  
// 000       000   000  000   000  
// 000       000000000  000000000  
// 000       000   000  000 0 000  
//  0000000  000   000  000   000  

void lookAtFrom(vec3 tgt, vec3 pos) 
{ 
    cam.tgt     = tgt;
    cam.pos     = pos;
    cam.pos2tgt = cam.tgt-cam.pos;
    cam.dir     = normalize(cam.pos2tgt);
    cam.x       = normalize(cross(cam.dir, vy));
    cam.up      = normalize(cross(cam.x,cam.dir));
    cam.dist    = length(cam.pos2tgt);
}
void lookAt  (vec3 tgt) { lookAtFrom(tgt, cam.pos); }
void lookFrom(vec3 pos) { lookAtFrom(cam.tgt, pos); }
void lookPan (vec3 pan) { lookAtFrom(cam.tgt+pan, cam.pos+pan); }
void lookPitch(float ang) { 
    cam.pos2tgt = rotAxisAngle(cam.pos2tgt, cam.x, ang); 
    cam.tgt     = cam.pos + cam.pos2tgt;
    cam.dir     = normalize(cam.pos2tgt);
    cam.up      = normalize(cross(cam.x,cam.dir));
}
void orbitPitch(float pitch)
{
    cam.pos2tgt = rotAxisAngle(cam.pos2tgt, cam.x, pitch); 
    cam.pos     = cam.tgt - cam.pos2tgt;
    cam.dir     = normalize(cam.pos2tgt);
    cam.up      = normalize(cross(cam.x,cam.dir));
}
void orbitYaw(float yaw)
{
    cam.pos2tgt = rotAxisAngle(cam.pos2tgt, vy, yaw); 
    cam.pos     = cam.tgt - cam.pos2tgt;
    cam.dir     = normalize(cam.pos2tgt);
    cam.x       = normalize(cross(cam.dir, vy));
    cam.up      = normalize(cross(cam.x,cam.dir));
}
void orbit(float pitch, float yaw) 
{
    orbitYaw(yaw);
    orbitPitch(pitch);
}

void initCam(float dist, vec2 rot)
{
    lookAtFrom(v0, rotAxisAngle(rotAxisAngle(vec3(0,0,-dist), -vx, 89.0*rot.y), vy, -90.0*rot.x));
    cam.fov = PI2; // 4.0;
}

// 00000000    0000000    0000000  000000000  
// 000   000  000   000  000          000     
// 00000000   000   000  0000000      000     
// 000        000   000       000     000     
// 000         0000000   0000000      000     

vec4 postProc(vec3 col, bool dither, bool gamma, bool vignette)
{
    if (dither)   col -= vec3(gradientNoise(gl.frag)/256.0); 
    if (gamma)    col  = pow(col, vec3(1.0/2.2));
    if (vignette) col *= vec3(smoothstep(1.8, 0.5, length(gl.uv)/max(gl.aspect,1.0)));
    return vec4(col, 1.0);
}
#define keys(x,y)  texelFetch(iChannel0, ivec2(x,y), 0)
bool keyState(int key) { return keys(key, 2).x < 0.5; }
bool keyDown(int key)  { return keys(key, 0).x > 0.5; }

#define ZERO min(iFrame,0)
#define CAM_DIST   6.0
#define MAX_STEPS  256
#define MIN_DIST   0.001
#define MAX_DIST   20.0
#define HEX_DIST   5.0

#define NONE 0
#define HEXA 1
#define GLOW 2

Mat[2] material = Mat[2](
    //  hue   sat  lum    shiny  glossy
    Mat(0.67, 1.0, 0.6,   0.3,   0.9 ),
    Mat(0.33, 1.0, 0.5,   0.1,   1.0 )
);

bool space, anim, soft, occl, light, dither, foggy, rotate, normal, depthb;

float hash(float n) { return fract(cos(n)*45758.5453); }
mat2  rot2(float a) { vec2 v = sin(vec2(1.570796, 0) + a); return mat2(v, -v.y, v.x); }

float at;
vec2 hexid;

// 000   000  00000000  000   000   0000000   
// 000   000  000        000 000   000   000  
// 000000000  0000000     00000    000000000  
// 000   000  000        000 000   000   000  
// 000   000  00000000  000   000  000   000  

float shoreDist(vec3 p)
{
    float ll = length(p.xz);
    return max(0.03+0.06*clamp01(p.y-3.0), ll-2.8);
}

float mountain(vec2 p)
{
    vec2 po = p-vec2(-0.5,0.5);
    return 0.1+0.6*dot(cos(po), cos(po))*(1.2-length(p)/2.6);
}

float hexHeight(vec2 p)
{
    float t = iTime*2.0;
    float shore = length(p);
    if (shore > 2.6) return 0.08 + sin(t+p.x)*0.02 + cos(t+p.y)*0.02;
    vec2 po = p-vec2(-0.5,0.5);
    return 0.15+(sin(iTime*1.2+PI/(2.0+length(p)))*0.5+0.75)*0.6*dot(cos(po), cos(po))*(1.2-shore/2.6);
}
 
// 00000000  000  00000000  000      0000000    
// 000       000  000       000      000   000  
// 000000    000  0000000   000      000   000  
// 000       000  000       000      000   000  
// 000       000  00000000  0000000  0000000    

void field()
{
    vec3 a = gl.sdf.pos;
    vec2 p = a.xz;
    
    float rd = 0.25;
    vec2 s = vec2(0.8660254, 1);

        
    float fy = 0.8660254;
    float ry = fy*rd;
    float rx = rd*1.5;
    float dy = ry*2.0;
    s = vec2(rd*1.5, dy);
    
    vec2 c1, c2, c3;
    vec2 fps = floor(p/s);
    bool odd = mod(fps.x,2.0) >= 1.0;
    if (odd) fps.y += 0.5;
    
    c1 = fps*s;
    
    if (odd)
    {
        c2 = c1+vec2(rx,-ry);
        c3 = c1+vec2(rx, ry);
    }
    else
    {
        c2 = c1+vec2( 0, dy);
        c3 = c1+vec2(rx, ry);
    }
    
    vec2  r1 = p - c1;
    vec2  r2 = p - c2;
    vec2  r3 = p - c3;
          
    float h1 = hexHeight(c1);
    float h2 = hexHeight(c2);
    float h3 = hexHeight(c3);
    
    float d1 = sdHexagon(vec3(r1.x,a.y-h1,r1.y), v0, vec3(rd*fy,h1,0.05));
    float d2 = sdHexagon(vec3(r2.x,a.y-h2,r2.y), v0, vec3(rd*fy,h2,0.05));
    float d3 = sdHexagon(vec3(r3.x,a.y-h3,r3.y), v0, vec3(rd*fy,h3,0.05));

    float d = min(min(d1,d2), d3);
    
    if (gl.march)
    {
        if      (d1 == d) hexid = c1;
        else if (d2 == d) hexid = c2;
        else if (d3 == d) hexid = c3;
    }
    
    sdMat(HEXA, d);
}

// 00     00   0000000   00000000   
// 000   000  000   000  000   000  
// 000000000  000000000  00000000   
// 000 0 000  000   000  000        
// 000   000  000   000  000        

float map(vec3 p)
{
    float t = sin(iTime)*0.5+0.5;
    
    gl.sdf = SDF(MAX_DIST, p, NONE);
    
    field();
    
    if (gl.march) { 
        sdMat(GLOW, sdSphere(gl.light3, 0.1));
        sdMat(GLOW, sdSphere(gl.light2, 0.1));
        sdMat(GLOW, sdSphere(gl.light1, 0.1));
    }
    
    return gl.sdf.dist;
}

// 00     00   0000000   00000000    0000000  000   000  
// 000   000  000   000  000   000  000       000   000  
// 000000000  000000000  0000000    000       000000000  
// 000 0 000  000   000  000   000  000       000   000  
// 000   000  000   000  000   000   0000000  000   000  

float march(in vec3 ro, in vec3 rd)
{
    float t = 0.0, d;
    for(int i = ZERO; i < MAX_STEPS; i++)
    {
        vec3 p = ro+rd*t;
        gl.rd = rd;
        d = map(p);
        t += min(d, shoreDist(p));
        if (d < MIN_DIST) return t;
        if (t > MAX_DIST) break;
    }
    gl.sdf.mat = NONE;
    return min(t, MAX_DIST);
}

vec3 getNormal(vec3 p)
{
    vec3 n = v0;
    for (int i=ZERO; i<4; i++) {
        vec3 e = 0.5773*(2.0*vec3((((i+3)>>1)&1),((i>>1)&1),(i&1))-1.0);
        n += e*map(p+e*0.0001); }
    return normalize(n);
}

//  0000000   00     00  0000000    000  00000000  000   000  000000000    
// 000   000  000   000  000   000  000  000       0000  000     000       
// 000000000  000000000  0000000    000  0000000   000 0 000     000       
// 000   000  000 0 000  000   000  000  000       000  0000     000       
// 000   000  000   000  0000000    000  00000000  000   000     000       

float getOcclusion(vec3 p, vec3 n)
{
    if (!occl) return 1.0;
    float a = 0.0;
    float weight = 1.0;
    for (int i = ZERO; i <= 6; i++)
    {
        float d = (float(i) / 6.0) * 0.3;
        a += weight * (d - map(p + n*d));
        weight *= 0.8;
    }
    float f = clamp01(1.0-a);
    return f*f;
}

// 000      000   0000000   000   000  000000000  
// 000      000  000        000   000     000     
// 000      000  000  0000  000000000     000     
// 000      000  000   000  000   000     000     
// 0000000  000   0000000   000   000     000     

vec3 getLight(vec3 p, vec3 n, int mat, float d)
{
    if (mat == NONE) return black;
    if (mat == GLOW) return white;
    
    Mat m = material[mat-1];

    vec3 bn = dither ? bumpMap(p, n, 0.003) : n;

    if (p.y > 0.25)
    {
        m.lum = 0.6+pow(mountain(hexid)*1.4, 2.0);
        m.hue = sin(iTime*0.1)*0.5+0.5;
    }
    
    vec3  col = hsl(m.hue, m.sat, m.lum);
    
    float dl1 = dot(bn,normalize(gl.light1-p));
    float dl2 = dot(bn,normalize(gl.light2-p));
    float dl3 = dot(bn,normalize(gl.light3-p));
    float dnl = max(max(dl1, dl2), dl3);
    
    col  = (light) ? gray(col) : col;
    
    col += pow(m.glossy, 3.0)*vec3(pow(smoothstep(0.0+m.glossy*0.9, 1.0, dnl), 1.0+40.0*m.glossy));
    col *= clamp(pow(dnl, 1.0+m.shiny*20.0), gl.ambient, 1.0) * getOcclusion(p, n);

    if (p.y < 0.25)
    {
        d = 1.0-length(hexid)/HEX_DIST;
        col *= smoothstep(0.04, 0.3, d);
        col *= clamp01(pow(p.y*4.0, 1.5));
    }
        
    return clamp01(col);
}

// 00     00   0000000   000  000   000  
// 000   000  000   000  000  0000  000  
// 000000000  000000000  000  000 0 000  
// 000 0 000  000   000  000  000  0000  
// 000   000  000   000  000  000   000  

void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    initGlobal(fragCoord, iResolution, iMouse, iTime);
    gl.zero = ZERO;
    gl.shadow = 0.5;
    for (int i = KEY_1; i <= KEY_9; i++) { if (keyDown(i)) { gl.option = i-KEY_1+1; break; } }
    
    rotate =  keyState(KEY_R);
    anim   =  keyState(KEY_RIGHT);
    occl   =  keyState(KEY_UP);
    dither =  keyState(KEY_D);
    normal = !keyState(KEY_X);
    depthb = !keyState(KEY_Z);
    light  = !keyState(KEY_LEFT);
    space  =  keyState(KEY_SPACE);
    foggy  =  keyState(KEY_F);
    
    if (anim) at = 0.5*iTime;
    
    initCam(CAM_DIST, vec2(0));
    
    lookAtFrom(vec3(0.5*0.25,-0.5*0.25,0), vec3(0,3.0,CAM_DIST));
    if (rotate)
        orbitYaw(-at*10.0);
            
    if (iMouse.z > 0.0)
        lookAtFrom(vec3(0.5*0.25,-0.5*0.25,0), rotAxisAngle(vec3(0,3.0,CAM_DIST-2.5*gl.mp.y), vy, gl.mp.x*90.0));
        
    #ifndef TOY
    if (space) lookAtFrom(iCenter, iCamera);
    #endif
    
    gl.uv = (2.0*fragCoord-iResolution.xy)/iResolution.y;
    vec3 rd = normalize(gl.uv.x*cam.x + gl.uv.y*cam.up + cam.fov*cam.dir);
    
    gl.light1 = vy*(1.5+0.7*(sin(iTime*1.2+PI/1.0))) - 2.25*vx - 1.3*vz;
    gl.light2 = vy*(1.5+0.7*(sin(iTime*1.2+PI/2.0))) + 1.14*vx - 1.08*vz;
    gl.light3 = vy*(1.5+0.7*(sin(iTime*1.2+PI/3.0))) + 1.14*vx + 1.98*vz;
    
    hexid = vec2(0);
    gl.march = true;
    float d = march(cam.pos, rd);
    gl.march = false;
    int mat = gl.sdf.mat;
    vec3  p = cam.pos + d * rd;
    vec3  n = getNormal(p);
    vec3  col = v0;
           
    if (normal || depthb)
    {
        vec3 nc = normal ? d >= MAX_DIST ? black : n : white;
        vec3 zc = depthb ? vec3(1.0-pow(d/MAX_DIST,0.1)) : white;
        col = nc*zc;
    }
    else
    {
        col = getLight(p, n, mat, d);
    }
        
    #ifndef TOY
    col += vec3(print(0,0,vec2(iFrameRate, iTime)));
    #endif    

    fragColor = vec4(sqrt(clamp(col, 0., 1.)), 1.0);
}


///////////////////////////////////////////////////////////////////////
// Shadertoy footer wrapper
///////////////////////////////////////////////////////////////////////

void main ( void )
{
	float alpha_on;
	float alpha_off;


	if (bgnum > 0){
		alpha_on  = scene_duration * (bgnum - 1.);
		alpha_off = scene_duration * (bgnum + 0.);
	}
	if (bgnum == 0){
		alpha_on  = scene_duration * (maxbgnum) - 1;
		alpha_off = scene_duration * (maxbgnum) + 1;
	}

	
	
	

	iTime = u_Elapsed * .5;


	if (iTime > 6000.){
		iTime = 6000. * fract(iTime / 6000.);
	}
	float cycle_time = maxbgnum * scene_duration;
	float cycles = (u_Elapsed)/cycle_time;
	float full_cycles = trunc(cycles);
    float part_cycles = u_Elapsed - (full_cycles * cycle_time);
	
	vec4 blank = vec4(0.);
	
	// Run the full program only at the right time..
	if ((part_cycles > alpha_on-1.)&&(part_cycles <= alpha_off+1.)){
		mainImage ( gl_FragColor, gl_FragCoord.xy );
		gl_FragColor.a = 1.0;
	}
	// Otherwise... just blank => massive performance boost
	else{
		gl_FragColor = blank;		
    }
	// Still need to fade it
	gl_FragColor.a = 0.0;
   if (part_cycles > alpha_on-1.){
	if (part_cycles <= alpha_off){
		gl_FragColor.a = 1.;
		if ((alpha_off - part_cycles)<1.){
			gl_FragColor.a = alpha_off - part_cycles;
		}
	}
   }
}


